/******************************************************************************
 * Copyright (c) 2004 palmOne, Inc. or its subsidiaries.
 * All rights reserved.
 *****************************************************************************/
/**
 * @ingroup Sound
 *
 */


/**
 *
 * @file 	PmSoundLibAudioGroup.h
 *
 * @brief	Public 68K include file used to associate sound channels with Audio
 * 			groups on Treo 600 and Treo 650 smartphones.
 *
 * This header file and associated header files
 * support the specific sound functionality of the Treo smartphones. You
 * should use the Palm OS Sound Manager APIs for most of your work.
 *
 * Notes:
 * The calling application should always load this library with
 * SysLibLoad() before use, even if it is already open by another
 * application(ie, SysLibFind() returns a valid refnum). When
 * the application is done with the library, it should be
 * unloaded with SysLibRemove(). We do this because there is
 * no good way to synchronize loading and unloading of libraries
 * among multiple applications. It also greatly simplifies internal
 * synchronization.
 *
 *   We define the concept of Audio Groups to classify sounds depending
 *   on the relative importance to the user:
 *
 *        pmAudioGroupApp    Audio normally generated by user applications
 *                           such as games, MP3 players, and others. This is the default
 *                           classification even if no audio group is assigned.
 *
 *        pmAudioGroupSystem Typical audio events generated by the system to
 *                           give audible feedback to the user about system events.
 *                           Most of the sounds produced through SndPlaySystemSound are
 *                           classified internally as System.
 *
 *        pmAudioGroupAlert  Alerts, Alarms, and Ringtones should be given this classification.
 *                           If you depend on the Attention Manager to make a sound, it will also
 *                           use this classification. The constant sndAlarm when calling
 *                           SndPlaySystemSound is classified as an Alert.
 *
 *        pmAudioGroupHighPriority
 *                           Sounds of great importance to the
 *                           user and that are routed with the highest priority.
 *                           We currently define hsSndBatLow and hsSndBatShutdown constants,
 *                           when calling SndPlaySystemSound, as high priority sounds.
 *
 *  If you depend on PalmOS Attention Manager for audio alerts, or use PalmOS
 *  SndPlaySystemSound, then the audio group is automatically assigned, and there
 *  is nothing else you need to do.
 *
 *  However, if you directly play audio events using other PalmOS APIs such
 *  as SndPlayResource, or SndStreamCreate, for example, then you may want
 *  to assign the audio group, unless the default pmAudioGroupApp is
 *  sufficient for your purposes.
 *
 *  To assign an audio group we attach that additional information
 *  to the PalmOS volume parameter associated with many of the Sound Manager
 *  API calls. We have found this is the most effective way to associate
 *  the audio group to a sound channel, and have that information filter down,
 *  through PalmOS, to our Audio Driver in the Hardware Abstraction Layer (HAL).
 *  In this way, there is no need to modify PalmOS, and we can allow a variety
 *  of components and applications to take advantage of the behavior and custom
 *  routing we assign to the different audio group classifications.
 *
 *  The drawback is that when the audio group is attached to the volume parameter,
 *  then the volume is limited to a maximum of 8K - 1, in steps of 8 (which
 *  represents 128 levels to unity). In contrast PalmOS 5.x volume parameter
 *  is defined with a maximum of 32K - 1, in steps of 1 (which represents
 *  1024 levels to unity). Our constrain is generally acceptable, since it
 *  represents granularity of about 1 dB step changes, with up to 8X amplification
 *  We currently use less precise granularity of > 10 db step changes, and a
 *  maximum of 6X amplification for the case of QCELP/AMR playback.
 *
 *  There is a risk that an existing application using the PalmOS API may
 *  apply a volume parameter that could be mistaken for a volume plus audio
 *  group designation, but this risk is fairly minimal as explained next.
 *
 *  The Audio Group uses a mark in the upper 3-bits of the volume parameter.
 *  If this mark is binary '011' which means a positive number >= 24K, then
 *  we will interpret the lower three bits as the audio group. The risk of
 *  mistaken interpretation from existing PalmOS applications will happen
 *  when the application request amplifications of 24X to 32X. In our
 *  experience we have Not seen any application going beyond the range of
 *  2X to 8X amplification. In case such extremely rare condition do
 *  occurs, the effect would be that the range 24X to 32X will be treated
 *  as the lower range of 0X to 8X, and there is a chance the sound will
 *  be routed/treated as other than application audio.
 *
 *  One more thing to note is that PalmOS allows negative numbers in the
 *  volume parameter to mean one of the usual volume parameters the
 *  user may adjust through the preferences panel: system volume, game
 *  volume, and alarm volume. In our current implementation we do NOT
 *  change these values, and we do NOT assign any audio group designation
 *  to these preferences driven volume settings. In other words, they
 *  will be treated as application audio group.
 *
 *  The use of Audio Group in a device will be present when the feature
 *  is registered with the PalmOS Feature Manager. This feature will
 *  be available when:
 *
 *     FtrGet( pmSoundLibCreatorID, pmSoundLibAudioGroupFtr, &value );
 *
 *  return no error, and the UInt32 value is NOT:
 *
 *     pmSoundLibAudioGroupFtr_Off
 *
 *  It is important to always check for this feature, since this is
 *  a custom enhancement which could be turned off for some future
 *  device, or platform.
 *
 *  To assign an audio group to your volume parameter you will invoke
 *  the function PmSndAddAudioGroupToVolume (currently implemented as
 *  a macro for simplicity). Here is an example on how to use this:
 *
 *      // Includes
 *      #include <FeatureMgr.h>
 *      #include <Common/Libraries/HsSoundLib/PmSoundLibAudioGroup.h>
 *
 *      // Declarations
 *      UInt32 value  = 0;    // temporary variable for FtrGet
 *      Int32  volume = 1024; // default max amplitude in PalmOS 5.x, can also use sndMaxAmp with earlier PalmOS
 *
 *      // Code Snippet
 *      if (   ( !FtrGet( pmSoundLibCreatorID, pmSoundLibAudioGroupFtr, &value ) ) // is the Audio Group feature supported?
 *          && ( value != pmSoundLibAudioGroupFtr_Off ) ) // is the Audio Group feature active?
 *      {
 *          volume = PmSndAddAudioGroupToVolume( volume, pmAudioGroupAlert ); // then, mark this sound as an alert
 *      }
 *      SndPlayResource( mySound, volume, sndFlagNormal );
 *
 *
 */


#ifndef PMSOUNDLIBAUDIOGROUP_H_
#define PMSOUNDLIBAUDIOGROUP_H_

/***********************************************************************
 * Palm OS common definitions
 ***********************************************************************/
#include <Common/Libraries/HsSoundLib/HsSoundLibCommon.h> /**<Included for hsSoundLibCreatorID. */

/**
 * @name PmSoundLibAudioGroup information
 *
 */
/*@{*/
#define pmSoundLibCreatorID         hsSoundLibCreatorID /**<Pass this as the creator ID used in FtrGet. */
#define pmSoundLibAudioGroupFtr     0x0001 /**<Use this as the featureNum passed to FtrGet. */
#define pmSoundLibAudioGroupFtr_Off 0      /**<Value returned in *valueP of FtrGet when Audio Group is disabled. */
#define pmSoundLibAudioGroupFtr_On  1      /**<Value returned in *valueP of FtrGet when Audio Group is enabled */
/*@}*/

#define PM_AUDIOGROUP_MASK     0xE007 /**<No definition. */

/**
 * Holds audio group information.
 */
typedef enum PmAudioGroupEnum
{
	pmAudioGroupNone         = 0x0000, /**<Default audio group when volume is less than 24K. */
	pmAudioGroupApp          = 0x6000, /**<Application audio, treated as pmAudioGroupNone, but group type saved. */
	pmAudioGroupSystem       = 0x6001, /**<System audio (as in SndPlaySystemSound). */
	pmAudioGroupAlert        = 0x6002, /**<Alert, alarms, ringtones that notify user of an event. */
	pmAudioGroupHighPriority = 0x6003, /**<Extremely important event such as low battery shutdown. */

	pmAudioGroupInteractive  = 0x6004, /**<Application audio, but for interactive use such as with games. */
	pmAudioGroupReserved1    = 0x6005, /**<For future use, DO NOT USE. */
	pmAudioGroupReserved2    = 0x6006, /**<For future use, DO NOT USE. */
	pmAudioGroupReserved3    = 0x6007  /**<For future use, DO NOT USE. */

} PmAudioGroupEnum;

/** @see PmAudioGroupEnum */
typedef UInt16 PmAudioGroup;


/**
 * @name Audio Group API functions
 *
 * Implemented as macros.
 */
/*@{*/
#define PmSndAddAudioGroupToVolume(volume,audioGroup)   (Int32)       (((!audioGroup) || (((Int32)(volume)) <= -1))? (volume) : ((((volume)+7) & ~PM_AUDIOGROUP_MASK) | (audioGroup)))
/**<Attaches an audio group designation to a PalmOS volume parameter.

 Use this macro if you want your sound channel to be designated differently
 than application sound.

 If this macro were implemented as a function, the prototype would be:

 Int32 //returned volume // PmSndAddAudioGroupToVolume  ( Int32 volume, PmAudioGroup audioGroup );
*/
#define PmSndRemoveAudioGroupFromVolume(volume)         (Int32)       ((((Int32)(volume)) < 0x6000)? (volume) : ((volume) & ~PM_AUDIOGROUP_MASK ))
/**<Removes the audio group designation from a PalmOS volume parameter.

 Use this if you previously attached a sound group designation and
 retrieve that volume again using PalmOS APIs. Or if you need to
 manipulate the overloaded volume parameter, as a volume only value.

 If this macro were implemented as a function, the prototype would be:

 Int32 //returned volume // PmSndRemoveAudioGroupFromVolume ( Int32 volume );
*/

#define PmSndGetAudioGroupFromVolume(volume)            (PmAudioGroup)((((Int32)(volume)) < 0x6000)? (pmAudioGroupNone) : ((volume) & PM_AUDIOGROUP_MASK))
/**<Removes the volume component of a PalmOS volume parameter, leaving
 a pure audio group component.

 Use this if you have already attached the audio group to a volume
 parameter and need to identify the audio group designation.

 If this macro were implemented as a function, the prototype would be:

 PmAudioGroup PmSndGetAudioGroupFromVolume ( Int32 volume );
*/
/*@}*/

#endif /* #ifndef PMSOUNDLIBAUDIOGROUP_H_ */
